import pandas as pd
import gzip
from pathlib import Path

def read_lookup_file(lookup_file):
    # Read the lookup file and return a set of values in the first column
    df = pd.read_csv(lookup_file, header=None)
    return set(df[0])

def process_tsv_gz_file(tsv_gz_file, lookup_values):
    # Decompress the .tsv.gz file and read it into a DataFrame
    with gzip.open(tsv_gz_file, 'rt') as f:
        df = pd.read_csv(f, sep='\t', header=None)

    # Ensure the DataFrame has the correct number of columns
    if df.shape[1] < 7:
        raise ValueError(f"File {tsv_gz_file} does not have 7 columns.")

    # Filter rows where col1 matches values in lookup_values
    filtered_df = df[df[1].isin(lookup_values)]
    
    # Add new columns
    filtered_df[8] = filtered_df[0].astype(str) + "_" + filtered_df[1].astype(str) + "_" + filtered_df[2].astype(str)
    filtered_df[9] = filtered_df[4] / filtered_df[5]

    # Convert Path to string and remove '.gz'
    output_file_path = str(tsv_gz_file).replace('.gz', '')
    
    # Save the processed DataFrame to a new .tsv file
    filtered_df.to_csv(output_file_path, sep='\t', index=False, header=False)

    print(f"Processed file saved as: {output_file_path}")

def main():
    # Define the path to the lookup file and working directory
    lookup_file = 'locALLforfirstfilter.csv'
    working_directory = Path('.')
    
    # Read the lookup values from the file
    lookup_values = read_lookup_file(lookup_file)

    # Process all .tsv.gz files in the working directory
    for file in working_directory.glob('*.tsv.gz'):
        process_tsv_gz_file(file, lookup_values)

if __name__ == "__main__":
    main()
